<?php
require_once "config.php";

/*
 * hourly_message.php
 * Sends hourly / promo message to all saved groups
 * Uses random user name from users.json
 */

// =======================================================
// 🔧 Telegram API helper using CURL
// =======================================================
function apiRequest($method, $params = []) {
    global $apiURL;
    $ch = curl_init($apiURL . $method);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => $params,
        CURLOPT_TIMEOUT        => 15,
    ]);
    $res = curl_exec($ch);
    if ($res === false) {
        $err = curl_error($ch);
        file_put_contents("status.log", "[".date("Y-m-d H:i:s")."] ❌ CURL error: $err\n", FILE_APPEND);
        return null;
    }
    file_put_contents("status.log", "[".date("Y-m-d H:i:s")."] $method → $res\n", FILE_APPEND);
    curl_close($ch);
    return json_decode($res, true);
}

// =======================================================
// 📂 Helpers to load JSON safely
// =======================================================
function loadJson($file) {
    if (!file_exists($file)) return [];
    $data = json_decode(file_get_contents($file), true);
    return is_array($data) ? $data : [];
}

function buildReplyMarkup($buttons) {
    // Auto-handle both flat and nested button arrays
    if (isset($buttons[0][0]) && is_array($buttons[0][0])) {
        return json_encode(['inline_keyboard' => $buttons]);
    }
    $rows = [];
    foreach ($buttons as $b) {
        $rows[] = [["text" => (string)$b['text'], "url" => (string)$b['url']]];
    }
    return json_encode(['inline_keyboard' => $rows]);
}

// =======================================================
// 🚀 Main logic
// =======================================================
$groups = loadJson("groups.json");
$users  = loadJson("users.json");

if (empty($groups)) {
    echo "⚠️ No groups saved.\n";
    exit;
}

foreach ($groups as $chat_id) {
    $displayName = "Member";

    // 🔹 Pick random real user from users.json for this group
    if (isset($users[$chat_id]) && !empty($users[$chat_id])) {
        $userList = array_values($users[$chat_id]);
        $randomUser = $userList[array_rand($userList)];

        $first = $randomUser['first_name'] ?? '';
        $last  = $randomUser['last_name']  ?? '';
        $displayName = trim($first . " " . $last);
        if ($displayName === "") $displayName = "Member";
    }

    // 💬 Replace {user} placeholder
    $msg = str_replace("{user}", htmlspecialchars($displayName), $GLOBALS['hourly_text']);
    $replyMarkup = buildReplyMarkup($GLOBALS['hourly_buttons']);

    // 🩶 Optional sticker before message (uncomment to enable)
    /*
    apiRequest("sendSticker", [
        "chat_id" => $chat_id,
        "sticker" => $GLOBALS['welcome_sticker_id']
    ]);
    */

    // 🚀 Send message
    $send = apiRequest("sendMessage", [
        "chat_id"                => $chat_id,
        "text"                   => $msg,
        "parse_mode"             => "HTML",
        "disable_web_page_preview" => true,
        "reply_markup"           => $replyMarkup
    ]);

    if (!empty($send['ok'])) {
        echo "✅ Sent to $chat_id as $displayName\n";
    } else {
        $err = isset($send['description']) ? $send['description'] : json_encode($send);
        echo "❌ Failed to send to $chat_id — $err\n";
        file_put_contents("status.log", "[".date("Y-m-d H:i:s")."] ❌ sendMessage failed for $chat_id — $err\n", FILE_APPEND);
    }

    // optional delay between groups to avoid rate limits
    usleep(500000); // 0.5 second pause
}

echo "✅ Done — check status.log for results\n";
?>