<?php
require_once "config.php";

// ============================================
// 🔧 Helper Functions
// ============================================
function apiRequest($method, $params = [])
{
    global $apiURL;
    $url = $apiURL . $method;
    $opts = [
        "http" => [
            "method"  => "POST",
            "header"  => "Content-Type: application/x-www-form-urlencoded\r\n",
            "content" => http_build_query($params)
        ]
    ];
    $ctx = stream_context_create($opts);
    $res = @file_get_contents($url, false, $ctx);
    if ($res === false) return null;
    return json_decode($res, true);
}

function loadJson($file)
{
    if (!file_exists($file)) return [];
    $data = json_decode(file_get_contents($file), true);
    return is_array($data) ? $data : [];
}

function saveJson($file, $data)
{
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function logStatus($text)
{
    file_put_contents("status.log", "[" . date("Y-m-d H:i:s") . "] " . $text . "\n", FILE_APPEND);
}

function makeKeyboard($buttons)
{
    $rows = [];
    foreach ($buttons as $b) {
        $rows[] = [["text" => $b["text"], "url" => $b["url"]]];
    }
    return ["inline_keyboard" => $rows];
}

// ============================================
// 🌐 Handle Webhook or Browser Visit
// ============================================
$input = file_get_contents("php://input");

// If opened via browser, show log
if (empty($input)) {
    echo "<h2>🤖 Telegram Bot Status</h2>";
    if (file_exists("status.log")) {
        echo "<pre style='background:#111;color:#0f0;padding:10px;border-radius:6px;'>";
        echo htmlspecialchars(file_get_contents("status.log"));
        echo "</pre>";
    } else {
        echo "No logs yet.";
    }
    exit;
}

$update = json_decode($input, true);

// ============================================
// 🧩 1️⃣ Save Group Chat ID
// ============================================
if (isset($update['message']['chat'])) {
    $chat = $update['message']['chat'];
    if (in_array($chat['type'], ['supergroup', 'group'])) {
        $chat_id = $chat['id'];
        $groups = loadJson("groups.json");
        if (!in_array($chat_id, $groups)) {
            $groups[] = $chat_id;
            saveJson("groups.json", $groups);
            logStatus("💾 Saved new group ID: $chat_id");
        }
    }
}

// ============================================
// 🧩 2️⃣ Save Users (senders + new members)
// ============================================
function saveUser($chat_id, $user)
{
    $file = "users.json";
    $data = file_exists($file) ? json_decode(file_get_contents($file), true) : [];
    if (!isset($data[$chat_id])) $data[$chat_id] = [];

    $uid = $user['id'];
    $first = $user['first_name'] ?? '';
    $last = $user['last_name'] ?? '';
    $username = isset($user['username']) ? '@' . $user['username'] : trim("$first $last");

    // save or update user
    $data[$chat_id][$uid] = [
        'id' => $uid,
        'first_name' => $first,
        'last_name' => $last,
        'username' => $username,
        'saved_at' => time()
    ];

    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    logStatus("👤 Saved user $username ($uid) in group $chat_id");
}

// Save user if they send a message
if (isset($update['message']['from'])) {
    $chat_id = $update['message']['chat']['id'];
    $user = $update['message']['from'];
    if (in_array($update['message']['chat']['type'], ['group', 'supergroup'])) {
        saveUser($chat_id, $user);
    }
}

// Save user if they join as new member
if (isset($update['message']['new_chat_members'])) {
    $chat_id = $update['message']['chat']['id'];
    $members = $update['message']['new_chat_members'];

    foreach ($members as $m) {
        if (!empty($m['is_bot'])) continue;

        // Save this member
        saveUser($chat_id, $m);

        // Send welcome sticker only (no text)
        apiRequest("sendSticker", [
            "chat_id" => $chat_id,
            "sticker" => $GLOBALS['welcome_sticker_id'],
            "reply_markup" => json_encode(makeKeyboard($GLOBALS['welcome_buttons']))
        ]);

        logStatus("👋 Sent welcome sticker in group $chat_id to user {$m['id']}");
    }
}
?>